/// @file eventHandler.cpp
///
/// @brief Implements the @ref EventHandler class
///
/// @component Uspi/DeviceDetector
///
/// @author F.Berat / ADITG/SWG / fberat@de.adit-jv.com
///
/// @copyright (c) 2016 Advanced Driver Information Technology.
/// This code is developed by Advanced Driver Information Technology.
/// Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
/// All rights reserved.
///
/// @see Udev

#include <iostream>
#include <thread>
#include <cstring>
#include <unistd.h>

#include <sys/epoll.h>

#include "logger.h"
#include "eventHandler.h"

namespace adit {
namespace uspi {

/// @defgroup EPOLL_DEFINES Definition used for epoll_wait function
/// @{
#define EPOLL_MAX_EVENTS 10     ///< The maximum amount of event to report
#define EPOLL_TIME_OUT   500    ///< The maximum time to wait for
/// @}

/// @brief Constructor
///
/// Creates the epoll file descriptor to be used and initializes members.
EventHandler::EventHandler():
    mEfd(-1),
    mTerminate(0)
{
    mEfd = epoll_create1(0);
}

/// @brief Desctructor
///
/// Closes the epoll file descriptor
EventHandler::~EventHandler()
{
    if (mEfd >= 0)
        close(mEfd);
}

/// @brief Wait for event on one of the registered file handler.
///
/// The function returns once one of the following happens:
/// - New event
/// - Error
/// - Timeout
///
/// Executed by the @ref eventLoop.
/// On event, the flags are checked, and the callback stored in data.ptr is
/// executed. Error may occurs if the flag are different from (EPOLLIN |
/// EPOLLET), if the callback is not set, or if it returns a negative value.
int EventHandler::eventWait(void)
{
    struct epoll_event *events;
    int i;
    int ret = 0;
    int n = 0;

    events = (struct epoll_event *)calloc(EPOLL_MAX_EVENTS, sizeof(*events));

    if (!events)
    {
        ddlog << kLogErr << "No memory available for events." << std::endl;
        return -1;
    }

    n = epoll_wait(mEfd, events, EPOLL_MAX_EVENTS, EPOLL_TIME_OUT);

    if (n < 0)
    {
        ddlog << kLogErr << "epoll_wait error: " << strerror(errno)
            << std::endl;

        free(events);

        if (errno == EINTR)
        {
            /* Only exit if the daemon has received QUIT/INT/TERM */
            return 0;
        }

        return -1;
    }

    for (i = 0 ; i < n ; i++)
    {
        int (*callback)() = (int (*)())events[i].data.ptr;

        if (!(events[i].events & (EPOLLIN | EPOLLET)))
        {
            ddlog << kLogErr << "Error while polling. Event received: 0x"
                << std::hex << events[i].events << std::endl;
            /* We only support one event producer.
             * Error means that this producer died.
             */
            ddlog << kLogErr << "Now exiting." << std::endl;
            ret = -1;
            break;
        }

        if (!callback)
        {
            ddlog << kLogErr << "Callback not found, exiting." << std::endl;
            ret = -1;
            break;
        }

        if (callback() < 0)
        {
            ddlog << "Error while calling the callback, exiting." << std::endl;
            ret = -1;
            break;
        }
    }

    free(events);
    return ret;

}

/// @brief The registration method.
/// @param fd The file descriptor to watch.
/// @param cb The function to execute on event.
///
/// Interface allowing clients to register a file descriptor and callback
/// to watch event out.
/// The callback address is copied in the event data pointer, it must therefore
/// be available when the event occurs.
int EventHandler::registerEvent(int fd, void *cb)
{
    struct epoll_event event;

    if ((fd < 0) || !cb)
    {
        ddlog << kLogWarning << "Wrong parameter to add event (" << fd
            << ", " << cb << ")" << std::endl;
        return -1;
    }

    ddlog << kLogDebug << "Setting up the event handler with ("
        << fd << ", " << cb << ")." << std::endl;

    event.data.ptr = cb;
    event.events = EPOLLIN | EPOLLET;

    return epoll_ctl(mEfd, EPOLL_CTL_ADD, fd, &event);
}

/// @brief The active loop.
/// Shall be executed if no thread is expected to be used.
void EventHandler::eventLoop(void)
{
    while (!mTerminate) {
        if (eventWait() != 0)
            break;
    }
}

/// @brief The thread creator
void EventHandler::startEventLoop(void)
{
    ddlog << kLogDebug << "Starting Event Loop" << std::endl;
    mEventThread = std::thread(&EventHandler::eventLoop, this);
}

/// @brief The waiter for the thread
void EventHandler::wait(void)
{
    mEventThread.join();
}

/// @brief The thread terminator
/// @param wait If the method must wait for the thread to terminate before
/// returning.
void EventHandler::terminateEventLoop(bool wait)
{
    ddlog << kLogDebug << "Terminating loop." << std::endl;

    mTerminate = 1;

    if (wait)
        this->wait();
}

} // namespace uspi
} // namespace adit
